<?php
// Security first - prevent direct access
if (!defined('IN_BOT')) {
    die('Direct access not permitted');
}

// Define debug mode
if (!defined('DEBUG_MODE')) {
    define('DEBUG_MODE', false);
}

require_once 'config.php';
require_once 'functions.php';

// Initialize game functions
global $game;
$game = new GameFunctions();

// Helper functions
function editMessage($chat_id, $message_id, $text, $keyboard = null) {
    global $game;
    return $game->editMessage($chat_id, $message_id, $text, $keyboard);
}

function createKeyboard($buttons) {
    return [
        'inline_keyboard' => $buttons
    ];
}

// Handle webhook update
function handleUpdate($update) {
    global $game;
    try {
        // Get user info
        $user = $update['message']['from'] ?? $update['callback_query']['from'];
        $telegram_id = $user['id'];
        $username = $user['username'] ?? 'Unknown';
        $first_name = $user['first_name'] ?? 'Unknown';
        
        // Register user if they don't exist
        if (!$game->getUser($telegram_id)) {
            $referrer = null;
            if (isset($update['message']['text']) && strpos($update['message']['text'], '/start') === 0) {
                $parts = explode(' ', $update['message']['text']);
                if (isset($parts[1])) {
                    $referrer = $parts[1];
                }
            }
            $game->registerUser($telegram_id, $username, $first_name, $referrer);
        }

        // Handle message
        if (isset($update['message'])) {
            $message = $update['message'];
            $chat_id = $message['chat']['id'];
            $text = $message['text'] ?? '';

            switch ($text) {
                case '/start':
                    $game->sendMenu($chat_id);
                    break;

                case '/balance':
                    $user = $game->getUser($telegram_id);
                    $stats = $game->getFarmStats($telegram_id);
                    $message = "💰 <b>あなたのステータス</b>\n\n";
                    $message .= "💰 現在の残高: $" . number_format($user['balance'], 2) . "\n";
                    $message .= "💰 毎時の収益: $" . number_format($stats['income_per_hour'], 2) . "\n";
                    $message .= "動物数: " . $stats['total_animals'] . "\n";
                    $message .= "クリック数: " . number_format($user['total_clicks']) . "\n";
                    $game->sendMessage($chat_id, $message);
                    break;

                default:
                    $game->sendMessage($chat_id, "不明なコマンドです。/startでメニューを表示します。\n\n使用可能なコマンド:\n/start - メニューを表示\n/balance - 残高を確認\n/daily - 毎日の報酬を受取\n/farm - 動物園を表示\n/buy - 動物を購入\n/withdraw - 出金を申請");
                    break;
            }
        }

        // Handle callback query
        if (isset($update['callback_query'])) {
            $callback = $update['callback_query'];
            $chat_id = $callback['message']['chat']['id'];
            $message_id = $callback['message']['message_id'];
            $data = $callback['data'];

            switch ($data) {
                case 'balance':
                    $user = $game->getUser($telegram_id);
                    $stats = $game->getFarmStats($telegram_id);
                    $message = "💰 <b>あなたのステータス</b>\n\n";
                    $message .= "💰 現在の残高: $" . number_format($user['balance'], 2) . "\n";
                    $message .= "💰 毎時の収益: $" . number_format($stats['income_per_hour'], 2) . "\n";
                    $message .= "動物数: " . $stats['total_animals'] . "\n";
                    $message .= "クリック数: " . number_format($user['total_clicks']);
                    editMessage($chat_id, $message_id, $message);
                    break;

                case 'farm':
                    $stats = $game->getFarmStats($telegram_id);
                    $message = "動物園 🐄\n\n";
                    $message .= "動物数: " . $stats['total_animals'] . "\n";
                    $message .= "毎時の収益: $" . number_format($stats['income_per_hour'], 2) . "\n\n";
                    
                    foreach ($stats['animals'] as $animal) {
                        $message .= "• " . htmlspecialchars($animal['name']) . ": " . $animal['quantity'] . "匹\n";
                        $message .= "  毎時の収益: $" . number_format($animal['income_per_hour'], 2) . "\n";
                    }
                    editMessage($chat_id, $message_id, $message);
                    break;

                case 'buy_animal':
                    $animals = $game->getAvailableAnimals();
                    $message = "動物を購入 🐣\n\n";
                    $buttons = [];
                    foreach ($animals as $animal) {
                        $message .= "• " . htmlspecialchars($animal['name']) . ": $" . number_format($animal['cost'], 2) . "\n";
                        $message .= "  毎時の収益: $" . number_format($animal['income_per_hour'], 2) . "\n";
                        $buttons[] = [['text' => $animal['name'], 'callback_data' => 'buy_' . $animal['id']]];
                    }
                    $keyboard = createKeyboard($buttons);
                    editMessage($chat_id, $message_id, $message, $keyboard);
                    break;

                case 'daily':
                    if ($game->canClaimDaily($telegram_id)) {
                        if ($game->claimDaily($telegram_id)) {
                            $user = $game->getUser($telegram_id);
                            $message = "🎁 毎日の報酬を受取りました！\n\n";
                            $message .= "新しい残高: $" . number_format($user['balance'], 2);
                            editMessage($chat_id, $message_id, $message);
                        }
                    } else {
                        editMessage($chat_id, $message_id, "申し訳ありませんが、もう報酬を受け取っています。\n明日またお試しください！");
                    }
                    break;

                case 'income':
                    $game->updateIncome($telegram_id);
                    $user = $game->getUser($telegram_id);
                    $message = "💰 毎時の収益を更新しました！\n\n";
                    $message .= "新しい収益: $" . number_format($user['income'], 2) . "\n\n";
                    $message .= "動物園を成長させ、収益を増やしましょう！";
                    editMessage($chat_id, $message_id, $message);
                    break;

                case 'referrals':
                    $referrals = $game->getUserReferrals($telegram_id);
                    $message = "リファラル 🤝\n\n";
                    $message .= "紹介数: " . count($referrals) . "\n\n";
                    foreach ($referrals as $referral) {
                        $message .= "• " . htmlspecialchars($referral['username']) . " (" . $referral['first_name'] . ")\n";
                    }
                    editMessage($chat_id, $message_id, $message);
                    break;

                case 'withdraw':
                    $user = $game->getUser($telegram_id);
                    $message = "Withdraw 💳\n\n";
                    $message .= "Current Balance: $" . number_format($user['balance'], 2) . "\n\n";
                    $message .= "Minimum Withdrawal: $" . MIN_WITHDRAWAL_AMOUNT . "\n\n";
                    $message .= "Withdrawal Fee: " . WITHDRAWAL_FEE_PERCENT . "%\n\n";
                    $message .= "Please select a withdrawal method:\n";
                    $buttons = [
                        [['text' => 'PayPal', 'callback_data' => 'withdraw_paypal']],
                        [['text' => 'Bitcoin', 'callback_data' => 'withdraw_bitcoin']],
                        [['text' => 'USDT', 'callback_data' => 'withdraw_usdt']]
                    ];
                    $keyboard = createKeyboard($buttons);
                    editMessage($chat_id, $message_id, $message, $keyboard);
                    break;

                case 'withdraw_paypal':
                case 'withdraw_bitcoin':
                case 'withdraw_usdt':
                    $payment_method = str_replace('withdraw_', '', $data);
                    $message = "出金 💳\n\n";
                    $message .= "選択した出金方法: " . ucfirst($payment_method) . "\n\n";
                    $message .= "出金先アドレスを入力してください:\n\n";
                    $message .= "例:\n";
                    $message .= "PayPal: your.email@example.com\n";
                    $message .= "Bitcoin: 1A1zP1eP5QGefi2DMPTfTL5SLmv7DivfNa\n";
                    $message .= "USDT: 0x1234567890abcdef...\n\n";
                    editMessage($chat_id, $message_id, $message);
                    break;

                default:
                    if (strpos($data, 'buy_') === 0) {
                        $animal_id = intval(substr($data, 4));
                        $animal = $game->getAvailableAnimals($animal_id);
                        if ($animal) {
                            $message = "動物を購入 🐣\n\n";
                            $message .= "選択した動物: " . htmlspecialchars($animal['name']) . "\n";
                            $message .= "価格: $" . number_format($animal['cost'], 2) . "\n";
                            $message .= "毎時の収益: $" . number_format($animal['income_per_hour'], 2) . "\n\n";
                            $message .= "購入数を選択してください:\n";
                            $buttons = [
                                [['text' => '1匹', 'callback_data' => 'buy_' . $animal_id . '_1']],
                                [['text' => '5匹', 'callback_data' => 'buy_' . $animal_id . '_5']],
                                [['text' => '10匹', 'callback_data' => 'buy_' . $animal_id . '_10']]
                            ];
                            $keyboard = createKeyboard($buttons);
                            editMessage($chat_id, $message_id, $message, $keyboard);
                        }
                    }
                    break;
            }
        }

    } catch (Exception $e) {
        error_log("Error handling update: " . $e->getMessage());
        if (isset($update['message'])) {
            $chat_id = $update['message']['chat']['id'];
            $message = "An error occurred. Please try again.\n\nIf the problem persists, please contact support.";
            if (defined('DEBUG_MODE') && DEBUG_MODE) {
                $message .= "\n\nError details: " . $e->getMessage();
            }
            global $game;
            $game->sendMessage($chat_id, $message);
        }
    }
}

// Main webhook handler
header('Content-Type: application/json');

try {
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        $raw_data = file_get_contents('php://input');
        $update = json_decode($raw_data, true);
        
        if ($update) {
            handleUpdate($update);
            echo json_encode(['ok' => true]);
        } else {
            error_log("Invalid JSON received");
            http_response_code(400);
            echo json_encode(['ok' => false, 'error' => 'Invalid JSON']);
        }
    } else {
        http_response_code(405);
        echo json_encode(['ok' => false, 'error' => 'Method not allowed']);
    }
} catch (Exception $e) {
    error_log("Webhook error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['ok' => false, 'error' => 'Internal server error']);
}
