<?php
// Security first - prevent direct access
if (!defined('IN_BOT')) {
    die('Direct access not permitted');
}

require_once 'db.php';

/**
 * Main game functions class
 * Handles all game logic and database interactions
 */
class GameFunctions {
    private $db;

    public function __construct() {
        $this->db = Database::getInstance();
        $this->db = $this->db->getConnection();
    }

    /**
     * Register a new user
     * @param int $telegram_id
     * @param string $username
     * @param string $first_name
     * @param int|null $referrer
     * @return bool
     */
    public function registerUser($telegram_id, $username, $first_name, $referrer = null) {
        try {
            $this->db->beginTransaction();
            
            // Check if user already exists
            $stmt = $this->db->query("SELECT telegram_id FROM users WHERE telegram_id = ?", [$telegram_id]);
            if ($stmt->fetch()) {
                return false;
            }

            // Insert new user
            $stmt = $this->db->query("INSERT INTO users (telegram_id, username, first_name, referral_id) VALUES (?, ?, ?, ?)", 
                [$telegram_id, $username, $first_name, $referrer]);

            // If user was referred, update referrer's count
            if ($referrer !== null) {
                $stmt = $this->db->query("UPDATE users SET referral_count = referral_count + 1 WHERE telegram_id = ?", [$referrer]);
            }

            $this->db->commit();
            return true;
        } catch (Exception $e) {
            $this->db->rollback();
            error_log("Error registering user: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Get user information
     * @param int $telegram_id
     * @return array|null
     */
    public function getUser($telegram_id) {
        try {
            $stmt = $this->db->query("SELECT * FROM users WHERE telegram_id = ?", [$telegram_id]);
            return $stmt->fetch();
        } catch (Exception $e) {
            error_log("Error getting user: " . $e->getMessage());
            return null;
        }
    }

    /**
     * Click to earn coins
     * @param int $telegram_id
     * @return bool
     */
    public function clickToEarn($telegram_id) {
        try {
            $user = $this->getUser($telegram_id);
            if (!$user) {
                return false;
            }

            // Check click cooldown
            $last_click = strtotime($user['last_click']);
            $now = time();
            if ($now - $last_click < CLICK_COOLDOWN) {
                return false;
            }

            // Update balance and last click
            $stmt = $this->db->query("UPDATE users SET 
                balance = balance + ?, 
                total_clicks = total_clicks + 1,
                last_click = CURRENT_TIMESTAMP 
                WHERE telegram_id = ?", 
                [CLICK_REWARD_AMOUNT, $telegram_id]);

            return $stmt->rowCount() > 0;
        } catch (Exception $e) {
            error_log("Error clicking to earn: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Buy an animal
     * @param int $telegram_id
     * @param int $animal_id
     * @param int $quantity
     * @return bool
     */
    public function buyAnimal($telegram_id, $animal_id, $quantity = 1) {
        try {
            $this->db->beginTransaction();

            // Get animal price
            $stmt = $this->db->query("SELECT cost FROM animals WHERE id = ?", [$animal_id]);
            $animal = $stmt->fetch();
            if (!$animal) {
                return false;
            }

            $total_cost = $animal['cost'] * $quantity;
            
            // Get user balance
            $stmt = $this->db->query("SELECT balance FROM users WHERE telegram_id = ?", [$telegram_id]);
            $user = $stmt->fetch();
            if (!$user || $user['balance'] < $total_cost) {
                return false;
            }

            // Update user balance
            $stmt = $this->db->query("UPDATE users SET balance = balance - ? WHERE telegram_id = ?", 
                [$total_cost, $telegram_id]);

            // Insert animal
            $stmt = $this->db->query("INSERT INTO user_animals (user_id, animal_id, quantity) VALUES (?, ?, ?)", 
                [$telegram_id, $animal_id, $quantity]);

            $this->db->commit();
            return true;
        } catch (Exception $e) {
            $this->db->rollback();
            error_log("Error buying animal: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Get farm statistics
     * @param int $telegram_id
     * @return array
     */
    public function getFarmStats($telegram_id) {
        try {
            // Get total animals
            $stmt = $this->db->query("SELECT SUM(quantity) as total_animals FROM user_animals WHERE user_id = ?", 
                [$telegram_id]);
            $total_animals = $stmt->fetch()['total_animals'] ?? 0;

            // Get income per hour
            $stmt = $this->db->query("SELECT SUM(ua.quantity * a.income_per_hour) as income_per_hour 
                FROM user_animals ua 
                JOIN animals a ON ua.animal_id = a.id 
                WHERE ua.user_id = ?", 
                [$telegram_id]);
            $income_per_hour = $stmt->fetch()['income_per_hour'] ?? 0;

            // Get animals breakdown
            $stmt = $this->db->query("SELECT a.name, ua.quantity, a.income_per_hour 
                FROM user_animals ua 
                JOIN animals a ON ua.animal_id = a.id 
                WHERE ua.user_id = ?", 
                [$telegram_id]);
            $animals = $stmt->fetchAll();

            return [
                'total_animals' => $total_animals,
                'income_per_hour' => $income_per_hour,
                'animals' => $animals
            ];
        } catch (Exception $e) {
            error_log("Error getting farm stats: " . $e->getMessage());
            return [
                'total_animals' => 0,
                'income_per_hour' => 0,
                'animals' => []
            ];
        }
    }

    /**
     * Get user referrals
     * @param int $telegram_id
     * @return array
     */
    public function getUserReferrals($telegram_id) {
        try {
            $stmt = $this->db->query("SELECT r.referred_id, u.username, u.first_name 
                FROM referrals r 
                JOIN users u ON r.referred_id = u.telegram_id 
                WHERE r.referrer_id = ?", 
                [$telegram_id]);
            return $stmt->fetchAll();
        } catch (Exception $e) {
            error_log("Error getting referrals: " . $e->getMessage());
            return [];
        }
    }

    /**
     * Process withdrawal request
     * @param int $telegram_id
     * @param float $amount
     * @param string $payment_method
     * @param string $payment_address
     * @return bool
     */
    public function requestWithdrawal($telegram_id, $amount, $payment_method, $payment_address) {
        try {
            $this->db->beginTransaction();

            // Check minimum withdrawal amount
            if ($amount < MIN_WITHDRAWAL_AMOUNT) {
                return false;
            }

            // Calculate fee
            $fee = $amount * (WITHDRAWAL_FEE_PERCENT / 100);
            $net_amount = $amount - $fee;

            // Check user balance
            $stmt = $this->db->query("SELECT balance FROM users WHERE telegram_id = ?", [$telegram_id]);
            $user = $stmt->fetch();
            if (!$user || $user['balance'] < $amount) {
                return false;
            }

            // Create withdrawal request
            $stmt = $this->db->query("INSERT INTO withdrawals (user_id, amount, payment_method, payment_address) 
                VALUES (?, ?, ?, ?)", 
                [$telegram_id, $net_amount, $payment_method, $payment_address]);

            // Update user balance
            $stmt = $this->db->query("UPDATE users SET balance = balance - ? WHERE telegram_id = ?", 
                [$amount, $telegram_id]);

            $this->db->commit();
            return true;
        } catch (Exception $e) {
            $this->db->rollback();
            error_log("Error processing withdrawal: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Check if user can claim daily reward
     * @param int $telegram_id
     * @return bool
     */
    public function canClaimDaily($telegram_id) {
        try {
            $stmt = $this->db->query("SELECT last_claim FROM users WHERE telegram_id = ?", [$telegram_id]);
            $user = $stmt->fetch();
            if (!$user) {
                return false;
            }

            $last_claim = strtotime($user['last_claim']);
            $now = time();
            return ($now - $last_claim) >= DAILY_CLAIM_INTERVAL;
        } catch (Exception $e) {
            error_log("Error checking daily claim: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Claim daily reward
     * @param int $telegram_id
     * @return bool
     */
    public function claimDaily($telegram_id) {
        try {
            if (!$this->canClaimDaily($telegram_id)) {
                return false;
            }

            $stmt = $this->db->query("UPDATE users SET 
                balance = balance + ?, 
                last_claim = CURRENT_TIMESTAMP 
                WHERE telegram_id = ?", 
                [DAILY_REWARD_AMOUNT, $telegram_id]);

            return $stmt->rowCount() > 0;
        } catch (Exception $e) {
            error_log("Error claiming daily reward: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Update user income
     * @param int $telegram_id
     * @return bool
     */
    public function updateIncome($telegram_id) {
        try {
            $stmt = $this->db->query("UPDATE users u 
                JOIN (
                    SELECT ua.user_id, SUM(ua.quantity * a.income_per_hour) as total_income
                    FROM user_animals ua
                    JOIN animals a ON ua.animal_id = a.id
                    GROUP BY ua.user_id
                ) sub ON u.telegram_id = sub.user_id
                SET u.income = sub.total_income
                WHERE u.telegram_id = ?", 
                [$telegram_id]);

            return $stmt->rowCount() > 0;
        } catch (Exception $e) {
            error_log("Error updating income: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Give daily bonus to user
     * @param int $telegram_id
     * @return bool
     */
    public function giveDailyBonus($telegram_id) {
        try {
            // Start transaction
            $this->db->beginTransaction();

            // Check if user can claim bonus
            $stmt = $this->db->prepare("SELECT last_bonus FROM users WHERE telegram_id = ?");
            $stmt->execute([$telegram_id]);
            $user = $stmt->fetch();
            
            if (!$user) {
                $this->sendMessage($telegram_id, "ユーザーが見つかりません。もう一度お試しください。\n\nエラー: ユーザーが存在しません");
                return false;
            }

            $last_bonus = strtotime($user['last_bonus']);
            $now = time();

            if (($now - $last_bonus) < 24 * 60 * 60) {
                $this->sendMessage($telegram_id, "申し訳ありませんが、もう報酬を受け取っています。\n明日またお試しください！");
                return false;
            }

            // Update balance and last_bonus
            $stmt = $this->db->prepare("
                UPDATE users 
                SET balance = balance + ?, 
                    last_bonus = NOW()
                WHERE telegram_id = ?
            ");
            $stmt->execute([DAILY_BONUS_AMOUNT, $telegram_id]);

            // Commit transaction
            $this->db->commit();

            // Get updated user info
            $user = $this->getUser($telegram_id);
            
            // Send success message
            $message = "🎁 毎日の報酬を受取りました！\n\n";
            $message .= "新しい残高: $" . number_format($user['balance'], 2);
            $this->sendMessage($telegram_id, $message);

            return true;
        } catch (Exception $e) {
            // Rollback transaction on error
            $this->db->rollBack();
            error_log("Error giving daily bonus: " . $e->getMessage());
            $this->sendMessage($telegram_id, "エラーが発生しました。もう一度お試しください。\n\nエラー: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Get available animals for purchase
     * @return array
     */
    public function getAvailableAnimals() {
        try {
            $stmt = $this->db->query("SELECT * FROM animals ORDER BY cost");
            return $stmt->fetchAll();
        } catch (Exception $e) {
            error_log("Error getting available animals: " . $e->getMessage());
            return [];
        }
    }

    /**
     * Get top players leaderboard
     * @return string Formatted leaderboard message
     */
    public function getTopPlayers() {
        try {
            $stmt = $this->db->query("
                SELECT telegram_id, username, first_name, balance,
                (SELECT COUNT(*) FROM user_animals WHERE user_animals.user_id = users.telegram_id) as total_animals
                FROM users 
                ORDER BY balance DESC 
                LIMIT 20
            ");
            
            $players = $stmt->fetchAll();
            
            $message = "🏆 <b>リーダーボード</b>\n\n";
            $message .= "現在のトップ20プレイヤー:\n\n";
            
            foreach ($players as $index => $player) {
                $rank = $index + 1;
                $username = $player['username'] ? '@' . htmlspecialchars($player['username']) : htmlspecialchars($player['first_name']);
                $message .= "<b>#$rank</b> $username\n";
                $message .= "💰 残高: $" . number_format($player['balance'], 2) . "\n";
                $message .= "動物数: " . $player['total_animals'] . "\n\n";
            }
            
            return $message;
        } catch (Exception $e) {
            error_log("Error getting top players: " . $e->getMessage());
            return "エラーが発生しました。リーダーボードを取得できませんでした。";
        }
    }

    /**
     * Send message via Telegram Bot API
     * @param int $chat_id
     * @param string $text
     * @param array|null $keyboard
     * @return bool
     */
    public function sendMessage($chat_id, $text, $keyboard = null) {
        try {
            $data = [
                'chat_id' => $chat_id,
                'text' => $text,
                'parse_mode' => 'HTML'
            ];
            
            if ($keyboard) {
                $data['reply_markup'] = $keyboard;
            }
            
            $ch = curl_init(TELEGRAM_API_URL . '/sendMessage');
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
            curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            $response = curl_exec($ch);
            curl_close($ch);
            
            return json_decode($response, true)['ok'] ?? false;
        } catch (Exception $e) {
            error_log("Error sending message: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Edit an existing message
     * @param int $chat_id
     * @param int $message_id
     * @param string $text
     * @param array|null $keyboard
     * @return bool
     */
    public function editMessage($chat_id, $message_id, $text, $keyboard = null) {
        try {
            $data = [
                'chat_id' => $chat_id,
                'message_id' => $message_id,
                'text' => $text,
                'parse_mode' => 'HTML'
            ];
            
            if ($keyboard) {
                $data['reply_markup'] = $keyboard;
            }
            
            $ch = curl_init(TELEGRAM_API_URL . '/editMessageText');
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
            curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            $response = curl_exec($ch);
            curl_close($ch);
            
            return json_decode($response, true)['ok'] ?? false;
        } catch (Exception $e) {
            error_log("Error editing message: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Send main menu with inline keyboard
     * @param int $chat_id
     * @return bool
     */
    public function sendMenu($chat_id) {
        $keyboard = [
            'inline_keyboard' => [
                [[
                    'text' => '🐾 Click to Earn',
                    'callback_data' => 'click'
                ]],
                [[
                    'text' => '🛒 Buy Animals',
                    'callback_data' => 'buy_animals'
                ]],
                [[
                    'text' => '📊 My Farm',
                    'callback_data' => 'my_farm'
                ]],
                [[
                    'text' => '👥 Referral',
                    'callback_data' => 'referral'
                ]],
                [[
                    'text' => '🎁 Daily Bonus',
                    'callback_data' => 'daily_bonus'
                ]],
                [[
                    'text' => '🏆 Leaderboard',
                    'callback_data' => 'leaderboard'
                ]]
            ]
        ];

        $message = "Welcome to Rich Animals Game!\n\n";
        $message .= "Select an option from the menu below:\n";
        $message .= "• Click to earn coins\n";
        $message .= "• Buy and breed animals\n";
        $message .= "• Check your farm status\n";
        $message .= "• Invite friends and earn rewards\n";
        $message .= "• Claim daily bonus\n";
        $message .= "• View leaderboard\n";

        return $this->sendMessage($chat_id, $message, $keyboard);
    }
}

// Create singleton instance
$game = new GameFunctions();
?>
