<?php
/**
 * Telegram Bot Installer
 * Web-based installation wizard
 */

session_start();

// Check if already installed
if (file_exists(__DIR__ . '/../config.php') && !isset($_GET['force'])) {
    $config = file_get_contents(__DIR__ . '/../config.php');
    if (strpos($config, 'INSTALLATION_COMPLETED') !== false) {
        die('Bot is already installed. To reinstall, add ?force to the URL.');
    }
}

// Installation steps
$steps = [
    1 => 'Welcome',
    2 => 'Requirements Check',
    3 => 'Database Setup',
    4 => 'Bot Configuration',
    5 => 'Admin Account',
    6 => 'Finalize'
];

// Get current step
$current_step = isset($_SESSION['install_step']) ? $_SESSION['install_step'] : 1;
if (isset($_POST['step'])) {
    $current_step = (int)$_POST['step'];
    $_SESSION['install_step'] = $current_step;
}

// Process form submissions
$error = $success = '';
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    switch($current_step) {
        case 2: // Requirements check acknowledgment
            if (isset($_POST['acknowledge'])) {
                $_SESSION['install_step'] = 3;
                header('Location: index.php');
                exit;
            }
            break;
            
        case 3: // Database setup
            if (isset($_POST['db_test'])) {
                // Test database connection
                $db_type = $_POST['db_type'];
                $db_host = $_POST['db_host'];
                $db_name = $_POST['db_name'];
                $db_user = $_POST['db_user'];
                $db_pass = $_POST['db_pass'];
                
                try {
                    if ($db_type === 'mysql') {
                        $pdo = new PDO(
                            "mysql:host=$db_host",
                            $db_user,
                            $db_pass
                        );
                        // Try to create database if it doesn't exist
                        $pdo->exec("CREATE DATABASE IF NOT EXISTS `$db_name`");
                        $pdo->exec("USE `$db_name`");
                    } else {
                        $db_file = __DIR__ . '/../' . $db_name;
                        $pdo = new PDO("sqlite:$db_file");
                    }
                    
                    $_SESSION['db_config'] = [
                        'type' => $db_type,
                        'host' => $db_host,
                        'name' => $db_name,
                        'user' => $db_user,
                        'pass' => $db_pass
                    ];
                    
                    $success = "Database connection successful!";
                    $_SESSION['install_step'] = 4;
                    header('Location: index.php');
                    exit;
                } catch (PDOException $e) {
                    $error = "Database connection failed: " . $e->getMessage();
                }
            }
            break;
            
        case 4: // Bot configuration
            if (isset($_POST['bot_token'])) {
                $bot_token = trim($_POST['bot_token']);
                $webhook_url = trim($_POST['webhook_url']);
                
                // Verify bot token
                $ch = curl_init("https://api.telegram.org/bot$bot_token/getMe");
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                $response = curl_exec($ch);
                $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                curl_close($ch);
                
                if ($http_code === 200) {
                    $result = json_decode($response, true);
                    if ($result && $result['ok']) {
                        $_SESSION['bot_config'] = [
                            'token' => $bot_token,
                            'webhook' => $webhook_url,
                            'username' => $result['result']['username']
                        ];
                        $_SESSION['install_step'] = 5;
                        header('Location: index.php');
                        exit;
                    } else {
                        $error = "Invalid bot token";
                    }
                } else {
                    $error = "Could not verify bot token";
                }
            }
            break;
            
        case 5: // Admin account
            if (isset($_POST['admin_username'])) {
                $admin_user = trim($_POST['admin_username']);
                $admin_pass = trim($_POST['admin_password']);
                $admin_telegram = trim($_POST['admin_telegram_id']);
                
                if (strlen($admin_pass) < 8) {
                    $error = "Password must be at least 8 characters";
                } else {
                    $_SESSION['admin_config'] = [
                        'username' => $admin_user,
                        'password' => password_hash($admin_pass, PASSWORD_DEFAULT),
                        'telegram_id' => $admin_telegram
                    ];
                    $_SESSION['install_step'] = 6;
                    header('Location: index.php');
                    exit;
                }
            }
            break;
            
        case 6: // Finalize installation
            if (isset($_POST['finalize'])) {
                try {
                    // Create configuration file
                    $config_template = file_get_contents(__DIR__ . '/templates/config.php');
                    $config = strtr($config_template, [
                        '{{DB_TYPE}}' => $_SESSION['db_config']['type'],
                        '{{DB_HOST}}' => $_SESSION['db_config']['host'],
                        '{{DB_NAME}}' => $_SESSION['db_config']['name'],
                        '{{DB_USER}}' => $_SESSION['db_config']['user'],
                        '{{DB_PASS}}' => $_SESSION['db_config']['pass'],
                        '{{BOT_TOKEN}}' => $_SESSION['bot_config']['token'],
                        '{{WEBHOOK_URL}}' => $_SESSION['bot_config']['webhook'],
                        '{{ADMIN_TELEGRAM_ID}}' => $_SESSION['admin_config']['telegram_id']
                    ]);
                    
                    file_put_contents(__DIR__ . '/../config.php', $config);
                    
                    // Initialize database
                    require_once __DIR__ . '/../config.php';
                    $pdo = getDBConnection();
                    
                    // Execute schema
                    if ($_SESSION['db_config']['type'] === 'mysql') {
                        $schema = file_get_contents(__DIR__ . '/../db.sql');
                    } else {
                        $schema = file_get_contents(__DIR__ . '/../schema_sqlite.sql');
                    }
                    $pdo->exec($schema);
                    
                    // Create admin account
                    $stmt = $pdo->prepare("
                        INSERT INTO admins (username, password, telegram_id, is_super_admin)
                        VALUES (?, ?, ?, 1)
                    ");
                    $stmt->execute([
                        $_SESSION['admin_config']['username'],
                        $_SESSION['admin_config']['password'],
                        $_SESSION['admin_config']['telegram_id']
                    ]);
                    
                    // Set webhook
                    $ch = curl_init("https://api.telegram.org/bot" . $_SESSION['bot_config']['token'] . "/setWebhook");
                    curl_setopt_array($ch, [
                        CURLOPT_POST => true,
                        CURLOPT_POSTFIELDS => ['url' => $_SESSION['bot_config']['webhook']],
                        CURLOPT_RETURNTRANSFER => true
                    ]);
                    $response = curl_exec($ch);
                    curl_close($ch);
                    
                    // Clear installation session
                    session_destroy();
                    
                    $success = "Installation completed successfully!";
                } catch (Exception $e) {
                    $error = "Installation failed: " . $e->getMessage();
                }
            }
            break;
    }
}

// Check system requirements
$requirements = [
    'php' => [
        'version' => '7.4.0',
        'current' => phpversion(),
        'status' => version_compare(phpversion(), '7.4.0', '>=')
    ],
    'extensions' => [
        'pdo' => extension_loaded('pdo'),
        'pdo_mysql' => extension_loaded('pdo_mysql'),
        'pdo_sqlite' => extension_loaded('pdo_sqlite'),
        'curl' => extension_loaded('curl'),
        'openssl' => extension_loaded('openssl')
    ],
    'writable' => [
        'config' => is_writable(__DIR__ . '/../') || is_writable(__DIR__ . '/../config.php'),
        'uploads' => is_writable(__DIR__ . '/../uploads') || mkdir(__DIR__ . '/../uploads', 0755, true)
    ]
];
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Telegram Bot Installer - Step <?php echo $current_step; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        body { background-color: #f8f9fa; }
        .wizard-container {
            max-width: 800px;
            margin: 50px auto;
            background: white;
            border-radius: 10px;
            box-shadow: 0 0 20px rgba(0,0,0,0.1);
            padding: 30px;
        }
        .step-indicator {
            margin-bottom: 30px;
            display: flex;
            justify-content: space-between;
        }
        .step {
            flex: 1;
            text-align: center;
            padding: 10px;
            position: relative;
        }
        .step:not(:last-child):after {
            content: '';
            position: absolute;
            top: 50%;
            right: -50%;
            width: 100%;
            height: 2px;
            background: #dee2e6;
            z-index: 1;
        }
        .step.active {
            color: #0d6efd;
            font-weight: bold;
        }
        .step.completed {
            color: #198754;
        }
        .step-number {
            width: 30px;
            height: 30px;
            border-radius: 50%;
            background: #dee2e6;
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 auto 10px;
            z-index: 2;
            position: relative;
        }
        .step.active .step-number {
            background: #0d6efd;
            color: white;
        }
        .step.completed .step-number {
            background: #198754;
            color: white;
        }
    </style>
</head>
<body>
    <div class="wizard-container">
        <!-- Step Indicator -->
        <div class="step-indicator">
            <?php foreach ($steps as $num => $name): ?>
                <div class="step <?php 
                    echo $num === $current_step ? 'active' : '';
                    echo $num < $current_step ? 'completed' : '';
                ?>">
                    <div class="step-number"><?php echo $num; ?></div>
                    <div class="step-name"><?php echo $name; ?></div>
                </div>
            <?php endforeach; ?>
        </div>

        <?php if ($error): ?>
            <div class="alert alert-danger"><?php echo htmlspecialchars($error); ?></div>
        <?php endif; ?>

        <?php if ($success): ?>
            <div class="alert alert-success"><?php echo htmlspecialchars($success); ?></div>
        <?php endif; ?>

        <!-- Step Content -->
        <?php switch($current_step): ?>
            
            <?php case 1: // Welcome ?>
                <h2 class="text-center mb-4">Welcome to the Bot Installer</h2>
                <div class="text-center mb-4">
                    <p class="lead">This wizard will help you set up your Telegram bot.</p>
                    <p>Make sure you have the following information ready:</p>
                    <ul class="list-unstyled">
                        <li>✓ Database credentials</li>
                        <li>✓ Telegram Bot Token (from @BotFather)</li>
                        <li>✓ Your server's domain name</li>
                        <li>✓ Admin Telegram ID</li>
                    </ul>
                </div>
                <form method="post">
                    <input type="hidden" name="step" value="2">
                    <div class="text-center">
                        <button type="submit" class="btn btn-primary btn-lg">Start Installation</button>
                    </div>
                </form>
                <?php break; ?>

            <?php case 2: // Requirements Check ?>
                <h2 class="mb-4">System Requirements Check</h2>
                
                <h5>PHP Version</h5>
                <ul class="list-group mb-4">
                    <li class="list-group-item d-flex justify-content-between align-items-center">
                        PHP <?php echo $requirements['php']['current']; ?>
                        <?php if ($requirements['php']['status']): ?>
                            <span class="badge bg-success">✓</span>
                        <?php else: ?>
                            <span class="badge bg-danger">✗</span>
                        <?php endif; ?>
                    </li>
                </ul>

                <h5>Required Extensions</h5>
                <ul class="list-group mb-4">
                    <?php foreach ($requirements['extensions'] as $ext => $installed): ?>
                        <li class="list-group-item d-flex justify-content-between align-items-center">
                            <?php echo $ext; ?>
                            <?php if ($installed): ?>
                                <span class="badge bg-success">✓</span>
                            <?php else: ?>
                                <span class="badge bg-danger">✗</span>
                            <?php endif; ?>
                        </li>
                    <?php endforeach; ?>
                </ul>

                <h5>File Permissions</h5>
                <ul class="list-group mb-4">
                    <?php foreach ($requirements['writable'] as $path => $writable): ?>
                        <li class="list-group-item d-flex justify-content-between align-items-center">
                            <?php echo $path; ?> directory
                            <?php if ($writable): ?>
                                <span class="badge bg-success">✓</span>
                            <?php else: ?>
                                <span class="badge bg-danger">✗</span>
                            <?php endif; ?>
                        </li>
                    <?php endforeach; ?>
                </ul>

                <form method="post">
                    <input type="hidden" name="step" value="2">
                    <div class="form-check mb-3">
                        <input class="form-check-input" type="checkbox" id="acknowledge" name="acknowledge" required>
                        <label class="form-check-label" for="acknowledge">
                            I acknowledge any missing requirements may affect the bot's functionality
                        </label>
                    </div>
                    <div class="d-flex justify-content-between">
                        <a href="?step=1" class="btn btn-secondary">Back</a>
                        <button type="submit" class="btn btn-primary">Continue</button>
                    </div>
                </form>
                <?php break; ?>

            <?php case 3: // Database Setup ?>
                <h2 class="mb-4">Database Configuration</h2>
                
                <form method="post">
                    <input type="hidden" name="step" value="3">
                    
                    <div class="mb-3">
                        <label class="form-label">Database Type</label>
                        <select name="db_type" class="form-select" id="db_type">
                            <option value="mysql">MySQL</option>
                            <option value="sqlite">SQLite</option>
                        </select>
                    </div>

                    <div id="mysql_fields">
                        <div class="mb-3">
                            <label class="form-label">Database Host</label>
                            <input type="text" class="form-control" name="db_host" value="localhost">
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Database Name</label>
                            <input type="text" class="form-control" name="db_name" value="telegram_bot">
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Database Username</label>
                            <input type="text" class="form-control" name="db_user">
                        </div>

                        <div class="mb-3">
                            <label class="form-label">Database Password</label>
                            <input type="password" class="form-control" name="db_pass">
                        </div>
                    </div>

                    <div id="sqlite_fields" style="display: none;">
                        <div class="mb-3">
                            <label class="form-label">Database File</label>
                            <input type="text" class="form-control" name="db_name" value="bot.db">
                            <div class="form-text">Relative to bot root directory</div>
                        </div>
                    </div>

                    <div class="d-flex justify-content-between">
                        <a href="?step=2" class="btn btn-secondary">Back</a>
                        <button type="submit" name="db_test" class="btn btn-primary">Test & Continue</button>
                    </div>
                </form>

                <script>
                    document.getElementById('db_type').addEventListener('change', function() {
                        const mysqlFields = document.getElementById('mysql_fields');
                        const sqliteFields = document.getElementById('sqlite_fields');
                        
                        if (this.value === 'mysql') {
                            mysqlFields.style.display = 'block';
                            sqliteFields.style.display = 'none';
                        } else {
                            mysqlFields.style.display = 'none';
                            sqliteFields.style.display = 'block';
                        }
                    });
                </script>
                <?php break; ?>

            <?php case 4: // Bot Configuration ?>
                <h2 class="mb-4">Bot Configuration</h2>
                
                <form method="post">
                    <input type="hidden" name="step" value="4">
                    
                    <div class="mb-3">
                        <label class="form-label">Bot Token</label>
                        <input type="text" class="form-control" name="bot_token" required
                               placeholder="123456789:ABCdefGHIjklMNOpqrsTUVwxyz">
                        <div class="form-text">
                            Get this from @BotFather on Telegram
                        </div>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Webhook URL</label>
                        <input type="url" class="form-control" name="webhook_url" required
                               placeholder="https://your-domain.com/bot.php">
                        <div class="form-text">
                            Must be HTTPS. Point to your bot.php file.
                        </div>
                    </div>

                    <div class="d-flex justify-content-between">
                        <a href="?step=3" class="btn btn-secondary">Back</a>
                        <button type="submit" class="btn btn-primary">Continue</button>
                    </div>
                </form>
                <?php break; ?>

            <?php case 5: // Admin Account ?>
                <h2 class="mb-4">Admin Account Setup</h2>
                
                <form method="post">
                    <input type="hidden" name="step" value="5">
                    
                    <div class="mb-3">
                        <label class="form-label">Admin Username</label>
                        <input type="text" class="form-control" name="admin_username" required>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Admin Password</label>
                        <input type="password" class="form-control" name="admin_password" required minlength="8">
                        <div class="form-text">At least 8 characters</div>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Admin Telegram ID</label>
                        <input type="text" class="form-control" name="admin_telegram_id" required>
                        <div class="form-text">
                            Send /my_id to @userinfobot to get your Telegram ID
                        </div>
                    </div>

                    <div class="d-flex justify-content-between">
                        <a href="?step=4" class="btn btn-secondary">Back</a>
                        <button type="submit" class="btn btn-primary">Continue</button>
                    </div>
                </form>
                <?php break; ?>

            <?php case 6: // Finalize ?>
                <h2 class="mb-4">Installation Summary</h2>
                
                <div class="mb-4">
                    <h5>Database Configuration</h5>
                    <ul class="list-group mb-3">
                        <li class="list-group-item">
                            <strong>Type:</strong> <?php echo $_SESSION['db_config']['type']; ?>
                        </li>
                        <?php if ($_SESSION['db_config']['type'] === 'mysql'): ?>
                            <li class="list-group-item">
                                <strong>Host:</strong> <?php echo $_SESSION['db_config']['host']; ?>
                            </li>
                        <?php endif; ?>
                        <li class="list-group-item">
                            <strong>Database:</strong> <?php echo $_SESSION['db_config']['name']; ?>
                        </li>
                    </ul>

                    <h5>Bot Configuration</h5>
                    <ul class="list-group mb-3">
                        <li class="list-group-item">
                            <strong>Username:</strong> @<?php echo $_SESSION['bot_config']['username']; ?>
                        </li>
                        <li class="list-group-item">
                            <strong>Webhook:</strong> <?php echo $_SESSION['bot_config']['webhook']; ?>
                        </li>
                    </ul>

                    <h5>Admin Account</h5>
                    <ul class="list-group mb-3">
                        <li class="list-group-item">
                            <strong>Username:</strong> <?php echo $_SESSION['admin_config']['username']; ?>
                        </li>
                        <li class="list-group-item">
                            <strong>Telegram ID:</strong> <?php echo $_SESSION['admin_config']['telegram_id']; ?>
                        </li>
                    </ul>
                </div>

                <form method="post">
                    <input type="hidden" name="step" value="6">
                    
                    <?php if ($success): ?>
                        <div class="text-center">
                            <p class="lead text-success">✅ Installation completed successfully!</p>
                            <a href="../web-admin/" class="btn btn-primary btn-lg">Go to Admin Panel</a>
                        </div>
                    <?php else: ?>
                        <div class="d-flex justify-content-between">
                            <a href="?step=5" class="btn btn-secondary">Back</a>
                            <button type="submit" name="finalize" class="btn btn-success">
                                Complete Installation
                            </button>
                        </div>
                    <?php endif; ?>
                </form>
                <?php break; ?>

        <?php endswitch; ?>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>