<?php
/**
 * Telegram Bot Installation Helper
 * Run this script to check system requirements and setup
 */

echo "<h1>🚀 Telegram Bot Installation Helper</h1>";

// Check PHP version
echo "<h2>📋 System Requirements Check</h2>";

$requirements = [
    'PHP Version >= 7.4' => version_compare(PHP_VERSION, '7.4.0', '>='),
    'PDO Extension' => extension_loaded('pdo'),
    'PDO MySQL Extension' => extension_loaded('pdo_mysql'),
    'cURL Extension' => extension_loaded('curl'),
    'JSON Extension' => extension_loaded('json'),
    'OpenSSL Extension' => extension_loaded('openssl'),
];

$all_good = true;
foreach ($requirements as $requirement => $status) {
    $color = $status ? 'green' : 'red';
    $icon = $status ? '✅' : '❌';
    echo "<p style='color: $color;'>$icon $requirement</p>";
    if (!$status) $all_good = false;
}

if (!$all_good) {
    echo "<p style='color: red;'><strong>❌ Some requirements are not met. Please install missing extensions.</strong></p>";
    exit;
}

echo "<p style='color: green;'><strong>✅ All requirements met!</strong></p>";

// Check file permissions
echo "<h2>📁 File Permissions Check</h2>";

$upload_dir = 'uploads/';
if (!file_exists($upload_dir)) {
    if (mkdir($upload_dir, 0755, true)) {
        echo "<p style='color: green;'>✅ Created uploads directory</p>";
    } else {
        echo "<p style='color: red;'>❌ Failed to create uploads directory</p>";
    }
} else {
    echo "<p style='color: green;'>✅ Uploads directory exists</p>";
}

if (is_writable($upload_dir)) {
    echo "<p style='color: green;'>✅ Uploads directory is writable</p>";
} else {
    echo "<p style='color: orange;'>⚠️ Uploads directory is not writable. Please set permissions to 755 or 777</p>";
}

// Check configuration
echo "<h2>⚙️ Configuration Check</h2>";

if (file_exists('config.php')) {
    require_once 'config.php';
    
    // Check bot token
    if (BOT_TOKEN === 'YOUR_BOT_TOKEN_HERE') {
        echo "<p style='color: red;'>❌ Bot token not configured</p>";
    } else {
        echo "<p style='color: green;'>✅ Bot token configured</p>";
    }
    
    // Check webhook URL
    if (WEBHOOK_URL === 'https://yourdomain.com/bot.php') {
        echo "<p style='color: red;'>❌ Webhook URL not configured</p>";
    } else {
        echo "<p style='color: green;'>✅ Webhook URL configured: " . WEBHOOK_URL . "</p>";
    }
    
    // Check database connection
    try {
        $pdo = getDBConnection();
        echo "<p style='color: green;'>✅ Database connection successful</p>";
        
        // Check if tables exist
        $tables = ['users', 'categories', 'messages', 'admins', 'message_logs', 'bot_sessions'];
        $existing_tables = [];
        
        foreach ($tables as $table) {
            $stmt = $pdo->query("SHOW TABLES LIKE '$table'");
            if ($stmt->rowCount() > 0) {
                $existing_tables[] = $table;
            }
        }
        
        if (count($existing_tables) === count($tables)) {
            echo "<p style='color: green;'>✅ All database tables exist</p>";
        } else {
            $missing = array_diff($tables, $existing_tables);
            echo "<p style='color: red;'>❌ Missing tables: " . implode(', ', $missing) . "</p>";
            echo "<p>Please import db.sql to create missing tables.</p>";
        }
        
        // Check admin users
        $stmt = $pdo->query("SELECT COUNT(*) FROM admins WHERE is_active = 1");
        $admin_count = $stmt->fetchColumn();
        
        if ($admin_count > 0) {
            echo "<p style='color: green;'>✅ Admin users configured ($admin_count active)</p>";
        } else {
            echo "<p style='color: red;'>❌ No active admin users found</p>";
        }
        
    } catch (Exception $e) {
        echo "<p style='color: red;'>❌ Database connection failed: " . $e->getMessage() . "</p>";
    }
    
} else {
    echo "<p style='color: red;'>❌ config.php file not found</p>";
}

// Check HTTPS
echo "<h2>🔒 Security Check</h2>";

if (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on') {
    echo "<p style='color: green;'>✅ HTTPS is enabled</p>";
} else {
    echo "<p style='color: orange;'>⚠️ HTTPS not detected. Telegram requires HTTPS for webhooks.</p>";
}

// Installation steps
echo "<h2>📝 Next Steps</h2>";
echo "<ol>";
echo "<li><strong>Configure Bot Token:</strong> Edit config.php and add your bot token from @BotFather</li>";
echo "<li><strong>Set Webhook URL:</strong> Update WEBHOOK_URL in config.php with your domain</li>";
echo "<li><strong>Database Setup:</strong> Import db.sql into your MySQL database</li>";
echo "<li><strong>Admin Configuration:</strong> Add your Telegram ID to ADMIN_IDS array</li>";
echo "<li><strong>Set Webhook:</strong> Run setWebhook.php to register your webhook with Telegram</li>";
echo "<li><strong>Test Bot:</strong> Send /start command to your bot</li>";
echo "<li><strong>Web Admin:</strong> Access web-admin/ directory to manage via web interface</li>";
echo "</ol>";

// Quick setup form
if (BOT_TOKEN === 'YOUR_BOT_TOKEN_HERE' || WEBHOOK_URL === 'https://yourdomain.com/bot.php') {
    echo "<h2>🛠️ Quick Configuration</h2>";
    echo "<p>Fill out the form below to automatically update your config.php:</p>";
    
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        $bot_token = $_POST['bot_token'] ?? '';
        $webhook_url = $_POST['webhook_url'] ?? '';
        $admin_id = $_POST['admin_id'] ?? '';
        
        if ($bot_token && $webhook_url && $admin_id) {
            $config_content = file_get_contents('config.php');
            
            $config_content = str_replace(
                "define('BOT_TOKEN', 'YOUR_BOT_TOKEN_HERE');",
                "define('BOT_TOKEN', '$bot_token');",
                $config_content
            );
            
            $config_content = str_replace(
                "define('WEBHOOK_URL', 'https://yourdomain.com/bot.php');",
                "define('WEBHOOK_URL', '$webhook_url');",
                $config_content
            );
            
            $config_content = str_replace(
                "123456789,  // Replace with your Telegram ID",
                "$admin_id,  // Your Telegram ID",
                $config_content
            );
            
            if (file_put_contents('config.php', $config_content)) {
                echo "<p style='color: green;'>✅ Configuration updated successfully!</p>";
                echo "<p><a href='setWebhook.php' class='btn btn-primary'>Set Webhook Now</a></p>";
            } else {
                echo "<p style='color: red;'>❌ Failed to update configuration. Please check file permissions.</p>";
            }
        }
    } else {
        $current_domain = $_SERVER['HTTP_HOST'] ?? 'yourdomain.com';
        $suggested_webhook = "https://$current_domain" . dirname($_SERVER['REQUEST_URI']) . "/bot.php";
        ?>
        <form method="POST" style="max-width: 500px; margin: 20px 0;">
            <div style="margin-bottom: 15px;">
                <label><strong>Bot Token:</strong></label><br>
                <input type="text" name="bot_token" style="width: 100%; padding: 8px;" 
                       placeholder="123456789:ABCdefGHIjklMNOpqrsTUVwxyz" required>
                <small>Get this from @BotFather on Telegram</small>
            </div>
            
            <div style="margin-bottom: 15px;">
                <label><strong>Webhook URL:</strong></label><br>
                <input type="url" name="webhook_url" style="width: 100%; padding: 8px;" 
                       value="<?php echo htmlspecialchars($suggested_webhook); ?>" required>
                <small>Your domain + /bot.php (must be HTTPS)</small>
            </div>
            
            <div style="margin-bottom: 15px;">
                <label><strong>Your Telegram ID:</strong></label><br>
                <input type="number" name="admin_id" style="width: 100%; padding: 8px;" 
                       placeholder="123456789" required>
                <small>Get this from @userinfobot on Telegram</small>
            </div>
            
            <button type="submit" style="background: #007bff; color: white; padding: 10px 20px; border: none; border-radius: 4px;">
                Update Configuration
            </button>
        </form>
        <?php
    }
}

echo "<h2>📚 Documentation</h2>";
echo "<p>For detailed setup instructions, please read <strong>README.md</strong></p>";

echo "<h2>🆘 Need Help?</h2>";
echo "<ul>";
echo "<li>Check the troubleshooting section in README.md</li>";
echo "<li>Verify all configuration values are correct</li>";
echo "<li>Check server error logs for detailed error messages</li>";
echo "<li>Ensure your domain has a valid SSL certificate</li>";
echo "</ul>";

echo "<hr>";
echo "<p><em>Installation helper completed. You can delete this file after setup.</em></p>";
?>
