<?php
/**
 * Telegram Message Sender Bot - Main Webhook Handler
 * Handles all incoming Telegram updates and admin commands
 */

require_once 'config.php';
require_once 'bot_functions.php';

// Get incoming update
$input = file_get_contents('php://input');
$update = json_decode($input, true);

// Log incoming updates for debugging
error_log("Bot Update: " . $input);

if (!$update) {
    http_response_code(200);
    exit;
}

// Handle different types of updates
if (isset($update['message'])) {
    handleMessage($update['message']);
} elseif (isset($update['callback_query'])) {
    handleCallbackQuery($update['callback_query']);
}

http_response_code(200);

/**
 * Handle incoming messages
 */
function handleMessage($message) {
    $chat_id = $message['chat']['id'];
    $telegram_id = $message['from']['id'];
    $username = $message['from']['username'] ?? '';
    $first_name = $message['from']['first_name'] ?? '';
    $last_name = $message['from']['last_name'] ?? '';
    $text = $message['text'] ?? '';
    
    // Register or update user
    registerUser($telegram_id, $username, $first_name, $last_name);
    
    // Handle commands
    if (strpos($text, '/') === 0) {
        handleCommand($chat_id, $telegram_id, $text);
    } else {
        // Handle admin conversation states
        if (isAdmin($telegram_id)) {
            handleAdminConversationEnhanced($chat_id, $telegram_id, $text, $message);
        } else {
            // Regular user message
            sendMessage($chat_id, "👋 Hello! I'm a message broadcasting bot. Stay tuned for updates!");
        }
    }
}

/**
 * Handle bot commands
 */
function handleCommand($chat_id, $telegram_id, $command) {
    switch ($command) {
        case '/start':
            handleStartCommand($chat_id, $telegram_id);
            break;
            
        case '/admin':
            if (isAdmin($telegram_id)) {
                showAdminMenu($chat_id);
            } else {
                sendMessage($chat_id, "❌ You are not authorized to use admin commands.");
            }
            break;
            
        case '/help':
            handleHelpCommand($chat_id, $telegram_id);
            break;
            
        case '/cancel':
            if (isAdmin($telegram_id)) {
                clearBotSession($telegram_id);
                sendMessage($chat_id, "❌ Operation cancelled.", getMainAdminKeyboard());
            }
            break;
            
        default:
            sendMessage($chat_id, "❓ Unknown command. Use /help for available commands.");
    }
}

/**
 * Handle /start command
 */
function handleStartCommand($chat_id, $telegram_id) {
    $user = getUser($telegram_id);
    
    if (isAdmin($telegram_id)) {
        $welcome_text = "🎛 <b>Admin Panel</b>\n\n";
        $welcome_text .= "Welcome back, Admin! You can manage the bot using the buttons below or use /admin command.\n\n";
        $welcome_text .= "👥 <b>User Stats:</b>\n";
        
        $stats = getUserStats();
        foreach ($stats as $stat) {
            $welcome_text .= "• {$stat['category_name']}: {$stat['active_users']} users\n";
        }
        
        sendMessage($chat_id, $welcome_text, getMainAdminKeyboard());
    } else {
        $welcome_text = "👋 <b>Welcome to our Bot!</b>\n\n";
        $welcome_text .= "Hello " . ($user['first_name'] ?? 'there') . "!\n\n";
        $welcome_text .= "You've been added to our notification system. ";
        $welcome_text .= "You'll receive important updates and announcements here.\n\n";
        $welcome_text .= "📂 Your category: <b>" . ($user['category_name'] ?? 'General') . "</b>\n\n";
        $welcome_text .= "Use /help to see available commands.";
        
        sendMessage($chat_id, $welcome_text);
    }
}

/**
 * Handle /help command
 */
function handleHelpCommand($chat_id, $telegram_id) {
    if (isAdmin($telegram_id)) {
        $help_text = "🎛 <b>Admin Commands:</b>\n\n";
        $help_text .= "/admin - Show admin menu\n";
        $help_text .= "/start - Show welcome message\n";
        $help_text .= "/cancel - Cancel current operation\n";
        $help_text .= "/help - Show this help message\n\n";
        $help_text .= "Use the inline buttons for easy navigation!";
    } else {
        $help_text = "ℹ️ <b>Available Commands:</b>\n\n";
        $help_text .= "/start - Welcome message\n";
        $help_text .= "/help - Show this help message\n\n";
        $help_text .= "Stay tuned for updates and announcements!";
    }
    
    sendMessage($chat_id, $help_text);
}

/**
 * Show main admin menu
 */
function showAdminMenu($chat_id) {
    $menu_text = "🎛 <b>Admin Control Panel</b>\n\n";
    $menu_text .= "Choose an action from the menu below:";
    
    sendMessage($chat_id, $menu_text, getMainAdminKeyboard());
}

/**
 * Get main admin keyboard
 */
function getMainAdminKeyboard() {
    return [
        'keyboard' => [
            [['text' => '📤 Send Message'], ['text' => '👥 Manage Users']],
            [['text' => '🗂 Categories'], ['text' => '📊 Statistics']],
            [['text' => '❌ Cancel']]
        ],
        'resize_keyboard' => true,
        'one_time_keyboard' => false
    ];
}

/**
 * Handle callback queries (inline button presses)
 */
function handleCallbackQuery($callback_query) {
    $chat_id = $callback_query['message']['chat']['id'];
    $message_id = $callback_query['message']['message_id'];
    $telegram_id = $callback_query['from']['id'];
    $data = $callback_query['data'];
    $callback_query_id = $callback_query['id'];
    
    // Answer callback query first
    answerCallbackQuery($callback_query_id);
    
    if (!isAdmin($telegram_id)) {
        answerCallbackQuery($callback_query_id, "❌ Unauthorized", true);
        return;
    }
    
    // Handle different callback data
    $parts = explode(':', $data);
    $action = $parts[0];
    
    switch ($action) {
        case 'send_message':
            // Handle inline send message callback
            showAdminMenu($chat_id);
            break;
            
        case 'select_category':
            // Handle category selection callback
            showAdminMenu($chat_id);
            break;
            
        case 'confirm_send':
            // Handle send confirmation callback
            showAdminMenu($chat_id);
            break;
            
        case 'back':
            showAdminMenu($chat_id);
            break;
    }
}

/**
 * Handle admin conversation states
 */
function handleAdminConversation($chat_id, $telegram_id, $text, $message) {
    $session = getBotSession($telegram_id);
    
    switch ($text) {
        case '📤 Send Message':
            startMessageComposer($chat_id, $telegram_id);
            break;
            
        case '👥 Manage Users':
            showUserManagement($chat_id);
            break;
            
        case '🗂 Categories':
            showCategoryManagement($chat_id);
            break;
            
        case '📊 Statistics':
            showStatistics($chat_id);
            break;
            
        case '❌ Cancel':
            clearBotSession($telegram_id);
            sendMessage($chat_id, "❌ Operation cancelled.", getMainAdminKeyboard());
            break;
            
        default:
            // Handle conversation states
            if (isset($session['state'])) {
                handleConversationState($chat_id, $telegram_id, $text, $message, $session);
            } else {
                sendMessage($chat_id, "❓ Please use the menu buttons or /admin command.");
            }
    }
}

/**
 * Start message composer
 */
function startMessageComposer($chat_id, $telegram_id) {
    $session = [
        'state' => 'composing_message',
        'step' => 'title',
        'message_data' => []
    ];
    
    saveBotSession($telegram_id, $session);
    
    $text = "✏️ <b>Message Composer</b>\n\n";
    $text .= "Let's create a new message to send to users.\n\n";
    $text .= "Step 1/5: Please enter the <b>title</b> for your message:";
    
    $keyboard = [
        'keyboard' => [
            [['text' => '❌ Cancel']]
        ],
        'resize_keyboard' => true,
        'one_time_keyboard' => false
    ];
    
    sendMessage($chat_id, $text, $keyboard);
}

/**
 * Handle conversation states
 */
function handleConversationState($chat_id, $telegram_id, $text, $message, $session) {
    if ($session['state'] === 'composing_message') {
        handleMessageComposerState($chat_id, $telegram_id, $text, $message, $session);
    }
}

/**
 * Handle message composer states
 */
function handleMessageComposerState($chat_id, $telegram_id, $text, $message, $session) {
    $step = $session['step'];
    $message_data = $session['message_data'];
    
    switch ($step) {
        case 'title':
            $message_data['title'] = $text;
            $session['message_data'] = $message_data;
            $session['step'] = 'description';
            saveBotSession($telegram_id, $session);
            
            sendMessage($chat_id, "✅ Title saved!\n\nStep 2/5: Please enter the <b>description</b> for your message:");
            break;
            
        case 'description':
            $message_data['description'] = $text;
            $session['message_data'] = $message_data;
            $session['step'] = 'image';
            saveBotSession($telegram_id, $session);
            
            $keyboard = [
                'keyboard' => [
                    [['text' => '⏭ Skip Image']],
                    [['text' => '❌ Cancel']]
                ],
                'resize_keyboard' => true
            ];
            
            sendMessage($chat_id, "✅ Description saved!\n\nStep 3/5: Please send an <b>image</b> for your message, or click 'Skip Image':", $keyboard);
            break;
            
        case 'image':
            if ($text === '⏭ Skip Image') {
                $message_data['image_url'] = '';
            } elseif (isset($message['photo'])) {
                // Get the largest photo
                $photos = $message['photo'];
                $largest_photo = end($photos);
                $message_data['image_url'] = $largest_photo['file_id'];
            } else {
                sendMessage($chat_id, "❌ Please send an image or click 'Skip Image'.");
                return;
            }
            
            $session['message_data'] = $message_data;
            $session['step'] = 'button_text';
            saveBotSession($telegram_id, $session);
            
            $keyboard = [
                'keyboard' => [
                    [['text' => '⏭ Skip Button']],
                    [['text' => '❌ Cancel']]
                ],
                'resize_keyboard' => true
            ];
            
            sendMessage($chat_id, "✅ Image saved!\n\nStep 4/5: Please enter the <b>button text</b>, or click 'Skip Button':", $keyboard);
            break;
            
        case 'button_text':
            if ($text === '⏭ Skip Button') {
                $message_data['button_text'] = '';
                $message_data['button_url'] = '';
                $session['message_data'] = $message_data;
                $session['step'] = 'target';
                saveBotSession($telegram_id, $session);
                
                showTargetSelection($chat_id);
            } else {
                $message_data['button_text'] = $text;
                $session['message_data'] = $message_data;
                $session['step'] = 'button_url';
                saveBotSession($telegram_id, $session);
                
                sendMessage($chat_id, "✅ Button text saved!\n\nStep 4b/5: Please enter the <b>button URL</b>:");
            }
            break;
            
        case 'button_url':
            if (filter_var($text, FILTER_VALIDATE_URL)) {
                $message_data['button_url'] = $text;
                $session['message_data'] = $message_data;
                $session['step'] = 'target';
                saveBotSession($telegram_id, $session);
                
                showTargetSelection($chat_id);
            } else {
                sendMessage($chat_id, "❌ Please enter a valid URL (starting with http:// or https://)");
            }
            break;
            
        case 'target':
            // This is handled by handleTargetSelection in the enhanced conversation handler
            break;
    }
}

/**
 * Show target selection (categories)
 */
function showTargetSelection($chat_id) {
    $categories = getCategories();
    $keyboard = [];
    
    // Add "Send to All" option
    $keyboard[] = [['text' => '📢 Send to All Users']];
    
    // Add category options
    foreach ($categories as $category) {
        $keyboard[] = [['text' => "📂 {$category['name']}"]];
    }
    
    $keyboard[] = [['text' => '❌ Cancel']];
    
    $reply_markup = [
        'keyboard' => $keyboard,
        'resize_keyboard' => true,
        'one_time_keyboard' => true
    ];
    
    sendMessage($chat_id, "🎯 <b>Step 5/5: Choose Target</b>\n\nWho should receive this message?", $reply_markup);
}

/**
 * Show user management
 */
function showUserManagement($chat_id) {
    $stats = getUserStats();
    $text = "👥 <b>User Management</b>\n\n";
    
    foreach ($stats as $stat) {
        $text .= "📂 <b>{$stat['category_name']}</b>: {$stat['active_users']} active users\n";
    }
    
    $text .= "\n💡 Use the web admin panel for detailed user management.";
    
    sendMessage($chat_id, $text, getMainAdminKeyboard());
}

/**
 * Show category management
 */
function showCategoryManagement($chat_id) {
    $categories = getCategories();
    $text = "🗂 <b>Categories</b>\n\n";
    
    foreach ($categories as $category) {
        $text .= "• {$category['name']}\n";
    }
    
    $text .= "\n💡 Use the web admin panel to add, edit, or delete categories.";
    
    sendMessage($chat_id, $text, getMainAdminKeyboard());
}

/**
 * Show statistics
 */
function showStatistics($chat_id) {
    $stats = getUserStats();
    $total_users = 0;
    
    $text = "📊 <b>Bot Statistics</b>\n\n";
    $text .= "👥 <b>Users by Category:</b>\n";
    
    foreach ($stats as $stat) {
        $text .= "• {$stat['category_name']}: {$stat['active_users']} users\n";
        $total_users += $stat['active_users'];
    }
    
    $text .= "\n📈 <b>Total Active Users:</b> {$total_users}\n";
    $text .= "📅 <b>Report Date:</b> " . date('Y-m-d H:i:s');
    
    sendMessage($chat_id, $text, getMainAdminKeyboard());
}

/**
 * Handle target selection and message sending
 */
function handleTargetSelection($chat_id, $telegram_id, $text) {
    $session = getBotSession($telegram_id);
    
    if (!isset($session['message_data'])) {
        sendMessage($chat_id, "❌ No message data found. Please start over.", getMainAdminKeyboard());
        clearBotSession($telegram_id);
        return;
    }
    
    $message_data = $session['message_data'];
    $target_category = null;
    $send_to_all = false;
    
    if ($text === '📢 Send to All Users') {
        $send_to_all = true;
    } else {
        // Extract category name from button text
        $category_name = str_replace('📂 ', '', $text);
        $categories = getCategories();
        
        foreach ($categories as $category) {
            if ($category['name'] === $category_name) {
                $target_category = $category['id'];
                break;
            }
        }
        
        if (!$target_category) {
            sendMessage($chat_id, "❌ Invalid category selected. Please try again.");
            return;
        }
    }
    
    // Show preview and confirmation
    showMessagePreview($chat_id, $telegram_id, $message_data, $target_category, $send_to_all);
}

/**
 * Show message preview and confirmation
 */
function showMessagePreview($chat_id, $telegram_id, $message_data, $target_category, $send_to_all) {
    $preview_text = "📋 <b>Message Preview</b>\n\n";
    
    // Show the formatted message
    if (!empty($message_data['title'])) {
        $preview_text .= "<b>" . htmlspecialchars($message_data['title']) . "</b>\n\n";
    }
    
    if (!empty($message_data['description'])) {
        $preview_text .= htmlspecialchars($message_data['description']) . "\n\n";
    }
    
    if (!empty($message_data['button_text'])) {
        $preview_text .= "🔗 Button: " . htmlspecialchars($message_data['button_text']) . "\n";
        $preview_text .= "🌐 URL: " . htmlspecialchars($message_data['button_url']) . "\n\n";
    }
    
    if (!empty($message_data['image_url'])) {
        $preview_text .= "🖼 Image: Attached\n\n";
    }
    
    // Show target info
    if ($send_to_all) {
        $users = getUsersByCategory();
        $preview_text .= "🎯 <b>Target:</b> All Users (" . count($users) . " users)\n\n";
    } else {
        $users = getUsersByCategory($target_category);
        $categories = getCategories();
        $category_name = '';
        foreach ($categories as $cat) {
            if ($cat['id'] == $target_category) {
                $category_name = $cat['name'];
                break;
            }
        }
        $preview_text .= "🎯 <b>Target:</b> " . $category_name . " (" . count($users) . " users)\n\n";
    }
    
    $preview_text .= "Are you sure you want to send this message?";
    
    // Save data for confirmation
    $session = [
        'state' => 'confirm_send',
        'message_data' => $message_data,
        'target_category' => $target_category,
        'send_to_all' => $send_to_all
    ];
    saveBotSession($telegram_id, $session);
    
    $keyboard = [
        'keyboard' => [
            [['text' => '✅ Yes, Send Message']],
            [['text' => '❌ Cancel']]
        ],
        'resize_keyboard' => true,
        'one_time_keyboard' => true
    ];
    
    sendMessage($chat_id, $preview_text, $keyboard);
}

/**
 * Handle message sending confirmation
 */
function handleSendConfirmation($chat_id, $telegram_id, $text) {
    if ($text !== '✅ Yes, Send Message') {
        clearBotSession($telegram_id);
        sendMessage($chat_id, "❌ Message sending cancelled.", getMainAdminKeyboard());
        return;
    }
    
    $session = getBotSession($telegram_id);
    
    if (!isset($session['message_data'])) {
        sendMessage($chat_id, "❌ No message data found. Please start over.", getMainAdminKeyboard());
        return;
    }
    
    $message_data = $session['message_data'];
    $target_category = $session['target_category'];
    $send_to_all = $session['send_to_all'];
    
    // Save message to database
    $message_id = saveMessage(
        $message_data['title'],
        $message_data['description'],
        $message_data['image_url'],
        $message_data['button_text'] ?? '',
        $message_data['button_url'] ?? '',
        $send_to_all ? null : $target_category,
        'telegram_admin'
    );
    
    if (!$message_id) {
        sendMessage($chat_id, "❌ Failed to save message. Please try again.", getMainAdminKeyboard());
        return;
    }
    
    $message_data['message_id'] = $message_id;
    
    // Send progress message
    sendMessage($chat_id, "📤 <b>Sending message...</b>\n\nPlease wait while we deliver your message to users.");
    
    // Send messages
    if ($send_to_all) {
        $result = sendToAll($message_data);
    } else {
        $result = sendToCategory($target_category, $message_data);
    }
    
    // Update message sent count
    try {
        $pdo = getDBConnection();
        $stmt = $pdo->prepare("UPDATE messages SET sent_count = ? WHERE id = ?");
        $stmt->execute([$result['sent'], $message_id]);
    } catch (Exception $e) {
        error_log("Error updating sent count: " . $e->getMessage());
    }
    
    // Send completion message
    $completion_text = "✅ <b>Message Sent Successfully!</b>\n\n";
    $completion_text .= "📊 <b>Delivery Report:</b>\n";
    $completion_text .= "✅ Sent: " . $result['sent'] . " users\n";
    $completion_text .= "❌ Failed: " . $result['failed'] . " users\n";
    $completion_text .= "📅 Time: " . date('Y-m-d H:i:s');
    
    sendMessage($chat_id, $completion_text, getMainAdminKeyboard());
    
    // Clear session
    clearBotSession($telegram_id);
}

/**
 * Enhanced admin conversation handler with target selection
 */
function handleAdminConversationEnhanced($chat_id, $telegram_id, $text, $message) {
    $session = getBotSession($telegram_id);
    
    // Handle target selection for message sending
    if (isset($session['state']) && $session['state'] === 'composing_message' && $session['step'] === 'target') {
        if (strpos($text, '📢') === 0 || strpos($text, '📂') === 0) {
            handleTargetSelection($chat_id, $telegram_id, $text);
            return;
        }
    }
    
    // Handle send confirmation
    if (isset($session['state']) && $session['state'] === 'confirm_send') {
        handleSendConfirmation($chat_id, $telegram_id, $text);
        return;
    }
    
    // Continue with regular conversation handling
    handleAdminConversation($chat_id, $telegram_id, $text, $message);
}
?>
