# 📱 Telegram Message Sender Bot

A complete Telegram bot system for broadcasting messages to users with both Telegram admin commands and a web-based admin panel.

## 🚀 Features

### Core Features
- **User Management**: Automatic user registration and categorization
- **Message Broadcasting**: Send formatted messages with images, titles, descriptions, and inline buttons
- **Multi-Category Support**: Organize users into different categories (Students, Clients, Members, etc.)
- **Dual Admin Interface**: Both Telegram bot commands and web admin panel
- **Message Tracking**: Complete delivery logs and statistics
- **Rich Message Format**: Support for images, formatted text, and inline buttons

### Telegram Bot Features
- ✅ Admin authentication via Telegram ID
- ✅ Interactive message composer with step-by-step guidance
- ✅ Category-based message targeting
- ✅ Message preview before sending
- ✅ Real-time delivery reports
- ✅ User statistics and management

### Web Admin Panel Features
- ✅ Secure login system
- ✅ Dashboard with user statistics
- ✅ Complete user management (view, edit, delete, change categories)
- ✅ Category management (add, edit, delete)
- ✅ Message composer with file upload
- ✅ Message delivery logs and analytics
- ✅ Responsive Bootstrap UI

## 📋 Requirements

- **PHP 7.4+** with PDO MySQL extension
- **MySQL 5.7+** or **MariaDB 10.2+**
- **Web server** (Apache/Nginx) with HTTPS support
- **cURL extension** enabled
- **Telegram Bot Token** from @BotFather

## 🛠️ Installation Guide

### Step 1: Create Telegram Bot

1. Open Telegram and search for `@BotFather`
2. Send `/newbot` command
3. Follow instructions to create your bot
4. Copy the bot token (format: `123456789:ABCdefGHIjklMNOpqrsTUVwxyz`)
5. Get your Telegram ID by messaging `@userinfobot`

### Step 2: Database Setup

1. Create a MySQL database:
```sql
CREATE DATABASE telegram_bot CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
```

2. Import the database structure:
```bash
mysql -u your_username -p telegram_bot < db.sql
```

3. Update the default admin password:
```sql
UPDATE admins SET password = '$2y$10$your_hashed_password' WHERE id = 1;
```

### Step 3: Configuration

1. Edit `config.php`:
```php
// Update these values
define('BOT_TOKEN', 'YOUR_BOT_TOKEN_HERE');
define('WEBHOOK_URL', 'https://yourdomain.com/bot.php');

// Database credentials
define('DB_HOST', 'localhost');
define('DB_NAME', 'telegram_bot');
define('DB_USER', 'your_db_username');
define('DB_PASS', 'your_db_password');

// Add your Telegram ID
define('ADMIN_IDS', [
    123456789,  // Your Telegram ID
]);

// Web admin credentials
define('WEB_ADMIN_USERNAME', 'admin');
define('WEB_ADMIN_PASSWORD', 'your_secure_password');
```

### Step 4: File Upload

1. Upload all files to your web server
2. Set proper permissions:
```bash
chmod 755 *.php
chmod 755 web-admin/
chmod 777 uploads/
```

3. Ensure HTTPS is enabled on your domain

### Step 5: Set Webhook

1. Open your browser and navigate to: `https://yourdomain.com/setWebhook.php`
2. Verify the webhook is set successfully
3. Test the bot by sending `/start` command

## 📁 Project Structure

```
telegram-bot/
├── config.php              # Configuration file
├── bot.php                 # Main webhook handler
├── bot_functions.php       # Bot utility functions
├── setWebhook.php         # Webhook setup script
├── db.sql                 # Database structure
├── uploads/               # Image upload directory
└── web-admin/             # Web admin panel
    ├── index.php          # Login page
    ├── dashboard.php      # Main dashboard
    ├── users.php          # User management
    ├── categories.php     # Category management
    ├── send_message.php   # Message composer
    ├── message_logs.php   # Delivery logs
    ├── auth_check.php     # Authentication middleware
    └── logout.php         # Logout handler
```

## 🎮 Usage Guide

### Telegram Bot Commands

**For Regular Users:**
- `/start` - Register and get welcome message
- `/help` - Show available commands

**For Admins:**
- `/admin` - Access admin control panel
- `/start` - View admin dashboard with statistics
- `/cancel` - Cancel current operation

### Admin Bot Interface

1. **Send Message Flow:**
   - Choose "📤 Send Message"
   - Enter title (step 1/5)
   - Enter description (step 2/5)
   - Upload image or skip (step 3/5)
   - Add button text or skip (step 4/5)
   - Add button URL if button text provided
   - Select target category or "Send to All"
   - Preview and confirm sending

2. **User Management:**
   - View user statistics by category
   - Access detailed management via web panel

### Web Admin Panel

1. **Login:** Navigate to `https://yourdomain.com/web-admin/`
2. **Dashboard:** View statistics and recent activity
3. **Users:** Manage users, change categories, toggle status
4. **Categories:** Add, edit, delete user categories
5. **Send Message:** Compose and send messages with file upload
6. **Logs:** View detailed delivery reports and analytics

## 🔧 Advanced Configuration

### Custom Categories

Add new categories via web admin panel or SQL:
```sql
INSERT INTO categories (name, description) VALUES ('VIP Members', 'Premium subscribers');
```

### Rate Limiting

Adjust message sending delays in `bot_functions.php`:
```php
usleep(100000); // 0.1 second delay between messages
```

### File Upload Settings

Modify upload limits in `config.php`:
```php
define('MAX_FILE_SIZE', 5 * 1024 * 1024); // 5MB
```

### Admin Access

Add more admin Telegram IDs:
```php
define('ADMIN_IDS', [
    123456789,  // Admin 1
    987654321,  // Admin 2
    555666777   // Admin 3
]);
```

## 🐛 Troubleshooting

### Common Issues

**Bot not responding:**
1. Check webhook URL is accessible
2. Verify bot token is correct
3. Check server error logs
4. Ensure HTTPS is working

**Database connection failed:**
1. Verify database credentials
2. Check if database exists
3. Ensure PDO MySQL extension is installed

**File upload not working:**
1. Check `uploads/` directory permissions (777)
2. Verify PHP file upload settings
3. Check disk space

**Messages not sending:**
1. Verify bot token and API access
2. Check rate limiting
3. Review message logs for errors

### Debug Mode

Enable error reporting in `config.php`:
```php
error_reporting(E_ALL);
ini_set('display_errors', 1);
```

### Log Files

Check these locations for error logs:
- `/var/log/apache2/error.log` (Apache)
- `/var/log/nginx/error.log` (Nginx)
- PHP error log (location varies)

## 🔒 Security Considerations

1. **Use HTTPS** for webhook URL
2. **Strong passwords** for admin accounts
3. **Regular backups** of database
4. **File upload validation** (implemented)
5. **SQL injection protection** (using prepared statements)
6. **XSS protection** (using htmlspecialchars)

## 📊 Database Schema

### Tables Overview

- **users**: Store Telegram user information
- **categories**: User categorization system
- **messages**: Sent message templates and logs
- **admins**: Web admin panel users
- **message_logs**: Individual delivery tracking
- **bot_sessions**: Temporary conversation states

### Key Relationships

- Users belong to categories (many-to-one)
- Messages can target specific categories
- Message logs track individual deliveries
- Bot sessions manage conversation flow

## 🚀 Deployment on Shared Hosting

### cPanel Instructions

1. **Upload Files:**
   - Use File Manager or FTP
   - Upload to `public_html` or subdirectory

2. **Database Setup:**
   - Create MySQL database via cPanel
   - Import `db.sql` via phpMyAdmin

3. **Configuration:**
   - Edit `config.php` with correct paths
   - Set file permissions via File Manager

4. **SSL Certificate:**
   - Enable SSL via cPanel
   - Update webhook URL to HTTPS

## 📈 Performance Optimization

### For High Volume

1. **Database Indexing:**
```sql
CREATE INDEX idx_users_telegram_id ON users(telegram_id);
CREATE INDEX idx_message_logs_status ON message_logs(status);
```

2. **Message Queue:**
   - Implement background job processing
   - Use Redis or database queue

3. **Caching:**
   - Cache user statistics
   - Store frequently accessed data

## 🤝 Contributing

1. Fork the repository
2. Create feature branch
3. Make changes with proper comments
4. Test thoroughly
5. Submit pull request

## 📄 License

This project is open source and available under the MIT License.

## 🆘 Support

For support and questions:
1. Check troubleshooting section
2. Review error logs
3. Test with minimal configuration
4. Create detailed issue report

---

**Made with ❤️ for the Telegram community**
